	function [N, Be, D, W] = L_PHASE_HP_FIR_ORDER(wT, d)
	
	%	Estimates the required order, N, for minimax designed linear-phase 
	%	highpass FIR filter. The estimate is better compared with remezord, 
	%	especially for cutoff edges close to 0 or  rad and long filters.
	%	L = filter length (No of samples in the impulse response) is L = N + 1.
	%
	%   inputs:
	%	  wT = [wsT wcT]
	%     wcT: passband cutoff edge (NORMALIZED 0 < wcT < )
	%     wsT: stopband cutoff edge < wcT 
	%	  d = [ds dc]
	%     dc: passband ripple 
	%     ds: stopband ripple < dc
	%
	%	outputs:
	%		Formated for direct use by PMR_FIR.m and firmp.m or remez.m
	
	% 	Toolbox for DIGITAL FILTERS USING MATLAB
	
	% Author: 			Lars Wanhammar 2004-07-17
	% Modified by: 	 	LW 2014-11-04
	% Copyright:		by authors - not released for commercial use
	% Version:			1 
	% Known bugs:		
	% Report bugs to:	Wanhammar@gmail.com
	%
	% References:		K. Ichige, M. Iwaki, and R. Ishii: Accurate Estimation of  
	%					Minimum Filter Length for Optimum FIR Digital Filters, 
	%					IEEE Trans. on Circuits and Systems - II, Vol. 47, No. 10, 
	%    				pp. 1008-1016, Oct. 2000.
	
	if ~(wT(2) > wT(1))
		error('Improper band edges. Edges should be: wsT < wcT')
	end			
	fc = wT(2)/pi/2;
	fs = wT(1)/pi/2;
	deltaF = fc - fs;	
	deltac = d(2);
	deltas = d(1);
	N = N4(0.5-fc, deltaF, deltac, deltas);
	Be = [0.0 wT(1) wT(2) pi]';
	D = [0 0 1 1]';
	W = [d(2)/d(1) 1]';
	
	%-----------------------------------------------------------
	function h = h13(fc, deltaF, c)
	h = (2/pi)*atan((c/deltaF)*(1/fc - 1/(0.5-deltaF)));
	%------------------------------------------------------------
	function g = g13(fc, deltaF, delta)
	v = 2.325*((-log10(delta))^-0.445)*deltaF^-1.39;
	g = (2/pi)*atan(v*(1/fc - 1/(0.5-deltaF)));
	%-----------------------------------------------------------
	function Nc = Nc(deltaF, delta)
	Nc = ceil((1.101/deltaF)*(-log10(2*delta))^1.1 +1);
	%-----------------------------------------------------------
	function N3 = N3(fc, deltaF, b)
	N3 = ceil(Nc(deltaF, b)*(g13(fc, deltaF, b)+g13(0.5-deltaF-fc, deltaF, b) + 1)/3);
	%-----------------------------------------------------------
	function Nm = Nm(deltaF, deltac, deltas)
	Nm = (0.52/deltaF)*log10(deltac/deltas)*((-log10(deltac))^0.17);
	%-----------------------------------------------------------
	function DN = DN(fc, deltaF, deltac, deltas)
	DN = ceil(Nm(deltaF, deltac, deltas)*(h13(fc, deltaF, 1.1)-0.5*(h13(0.5-deltaF-fc, deltaF, 0.29)-1)));
	%-----------------------------------------------------------
	function N4 = N4(fc, deltaF, deltac, deltas)
	N4 = N3(fc, deltaF, deltac) + DN(fc, deltaF, deltac, deltas);
	%-----------------------------------------------------------
	
	
